<?php
/**
 * tastyc functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package tastyc
 */

define( 'TASTYC_EXTRA_PLUGINS_DIRECTORY', 'https://bslthemes.com/plugins-latest/tastyc/' );
define( 'TASTYC_EXTRA_PLUGINS_PREFIX', 'Tastyc' );

if ( ! function_exists( 'tastyc_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function tastyc_setup() {
		/*
		 * Make theme available for translation.
		 * Translations can be filed in the /languages/ directory.
		 * If you're building a theme based on tastyc, use a find and replace
		 * to change 'tastyc' to the name of your theme in all the template files.
		 */
		load_theme_textdomain( 'tastyc', get_template_directory() . '/languages' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		/*
		 * Let WordPress manage the document title.
		 * By adding theme support, we declare that this theme does not use a
		 * hard-coded <title> tag in the document head, and expect WordPress to
		 * provide it for us.
		 */
		add_theme_support( 'title-tag' );

		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 *
		 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		 */
		add_theme_support( 'post-thumbnails' );

		// This theme uses wp_nav_menu() in one location.
		register_nav_menus( array(
			'primary' => esc_html__( 'Primary Menu', 'tastyc' ),
		) );

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support( 'html5', array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		) );

		// Add theme support for selective refresh for widgets.
		add_theme_support( 'customize-selective-refresh-widgets' );

		// Image Sizes
		add_image_size( 'tastyc_140x140', 280, 280, true );
		add_image_size( 'tastyc_256x256', 512, 512, true );
		add_image_size( 'tastyc_700x456', 700, 456, true );
		add_image_size( 'tastyc_950xAuto', 950, 9999, false );
		add_image_size( 'tastyc_1280x768', 1280, 768, true );
		add_image_size( 'tastyc_1920xAuto', 1920, 9999, false );
	}
endif;
add_action( 'after_setup_theme', 'tastyc_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function tastyc_content_width() {
	// This variable is intended to be overruled from themes.
	// Open WPCS issue: {@link https://github.com/WordPress-Coding-Standards/WordPress-Coding-Standards/issues/1043}.
	// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
	$GLOBALS['content_width'] = apply_filters( 'tastyc_content_width', 900 );
}
add_action( 'after_setup_theme', 'tastyc_content_width', 0 );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function tastyc_widgets_init() {
	register_sidebar( array(
		'name'		  => esc_html__( 'Sidebar', 'tastyc' ),
		'id'			=> 'sidebar-1',
		'description'   => esc_html__( 'Add widgets here.', 'tastyc' ),
		'before_widget' => '<section id="%1$s" class="widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	if ( class_exists( 'WooCommerce' ) ) :
		register_sidebar( array(
			'name'          => esc_html__( 'Shop Sidebar', 'tastyc' ),
			'id'            => 'shop-sidebar',
			'description'   => esc_html__( 'Sidebar that appears on the shop.', 'tastyc' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h1 class="widget-title">',
			'after_title'   => '</h1>',
		) );
	endif;
}
add_action( 'widgets_init', 'tastyc_widgets_init' );

/**
 * Register Default Fonts
 */
function tastyc_primary_fonts_url() {
	$fonts_url = '';

	/* Translators: If there are characters in your language that are not
	 * supported by Lora, translate this to 'off'. Do not translate
	 * into your own language.
	 */
	$josefin = _x( 'on', 'Josefin: on or off', 'tastyc' );

	if ( 'off' !== $josefin ) {
		$font_families = array();

		$font_families[] = 'Josefin+Sans:ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;1,100;1,200;1,300;1,400;1,500;1,600;1,700';

		$query_args = array(
			'family' => implode( '&family=', $font_families ),
			'display' => 'swap',
		);

		$fonts_url = add_query_arg( $query_args, 'https://fonts.googleapis.com/css2' );
	}

	return $fonts_url;
}

function tastyc_secondary_fonts_url() {
	$fonts_url = '';

	/* Translators: If there are characters in your language that are not
	 * supported by Lora, translate this to 'off'. Do not translate
	 * into your own language.
	 */
	$playfair = _x( 'on', 'Playfair Display: on or off', 'tastyc' );

	if ( 'off' !== $playfair ) {
		$font_families = array();

		$font_families[] = 'Playfair+Display:ital,wght@0,400;0,500;0,600;0,700;0,800;0,900;1,400;1,500;1,600;1,700;1,800;1,900';

		$query_args = array(
			'family' => implode( '&family=', $font_families ),
			'display' => 'swap',
		);

		$fonts_url = add_query_arg( $query_args, 'https://fonts.googleapis.com/css2' );
	}

	return $fonts_url;
}

/**
 * Enqueue scripts and styles.
 */
function tastyc_stylesheets() {
	// Web fonts
	wp_enqueue_style( 'tastyc-primary-fonts', tastyc_primary_fonts_url(), array(), null );
	wp_enqueue_style( 'tastyc-secondary-fonts', tastyc_secondary_fonts_url(), array(), null );

	$headingsFont =  get_field( 'heading_font_family', 'options' );
	$paragraphsFont =  get_field( 'text_font_family', 'options' );
	$buttonsFont =  get_field( 'btn_font_family', 'options' );

	// Custom fonts
	if ( $headingsFont ) {
		wp_enqueue_style( 'tastyc-heading-font', $headingsFont['url'] , array(), null );
	}
	if ( $paragraphsFont ) {
		wp_enqueue_style( 'tastyc-paragraph-font', $paragraphsFont['url'] , array(), null );
	}
	if ( $buttonsFont ) {
		wp_enqueue_style( 'tastyc-btn-font', $buttonsFont['url'] , array(), null );
	}

	// Custom Layouts
	$ui_skin = get_field( 'ui_skin', 'options' );
	$ui_type = get_field( 'ui_type', 'options' );

	/*Styles*/
	wp_enqueue_style( 'bootstrap', get_template_directory_uri() . '/assets/css/bootstrap.css', '1.0' );
	wp_enqueue_style( 'fontawesome', get_template_directory_uri() . '/assets/css/font-awesome.css', '1.0' );
	wp_enqueue_style( 'datepicker', get_template_directory_uri() . '/assets/css/datepicker.css', '1.0' );
	wp_enqueue_style( 'swiper', get_template_directory_uri() . '/assets/css/swiper.css', '1.0' );
	wp_enqueue_style( 'nice-select', get_template_directory_uri() . '/assets/css/nice-select.css', '1.0' );
	wp_enqueue_style( 'tastyc-mapbox', get_template_directory_uri() . '/assets/css/mapbox.css', '1.0' );
	wp_enqueue_style( 'magnific-popup', get_template_directory_uri() . '/assets/css/magnific-popup.css', '1.0' );
	wp_enqueue_style( 'photoswipe', get_template_directory_uri() . '/assets/css/photoswipe.css', '1.0' );
	wp_enqueue_style( 'tastyc-style', get_stylesheet_uri(), array( 'bootstrap', 'fontawesome', 'datepicker', 'swiper', 'tastyc-mapbox', 'nice-select', 'magnific-popup', 'photoswipe' ) );

	if ( $ui_skin == 1 && $ui_type == 0 ) {
		wp_enqueue_style( 'tastyc-dark', get_template_directory_uri() . '/assets/css/dark.css', array( 'tastyc-style' ), '1.0' );
	}
	if ( $ui_skin == 1 && $ui_type == 1 ) {
		wp_enqueue_style( 'tastyc-dark-minimal', get_template_directory_uri() . '/assets/css/dark-minimal.css', array( 'tastyc-style' ), '1.0' );
	}
}
add_action( 'wp_enqueue_scripts', 'tastyc_stylesheets' );

function tastyc_scripts() {

	$monthNames = array( esc_attr__( 'January', 'tastyc' ), esc_attr__( 'February', 'tastyc' ), esc_attr__( 'March', 'tastyc' ), esc_attr__( 'April', 'tastyc' ), esc_attr__( 'May', 'tastyc' ), esc_attr__( 'June', 'tastyc' ), esc_attr__( 'July', 'tastyc' ), esc_attr__( 'August', 'tastyc' ), esc_attr__( 'September', 'tastyc' ), esc_attr__( 'October', 'tastyc' ), esc_attr__( 'November', 'tastyc' ), esc_attr__( 'December', 'tastyc' ) );
	$monthNamesShort = array( esc_attr__( 'Jan', 'tastyc' ), esc_attr__( 'Feb', 'tastyc' ), esc_attr__( 'Mar', 'tastyc' ), esc_attr__( 'Apr', 'tastyc' ), esc_attr__( 'May', 'tastyc' ), esc_attr__( 'Jun', 'tastyc' ), esc_attr__( 'Jul', 'tastyc' ), esc_attr__( 'Aug', 'tastyc' ), esc_attr__( 'Sep', 'tastyc' ), esc_attr__( 'Oct', 'tastyc' ), esc_attr__( 'Nov', 'tastyc' ), esc_attr__( 'Dec', 'tastyc' ) );
	$dayNames = array( esc_attr__( 'Sunday', 'tastyc' ), esc_attr__( 'Monday', 'tastyc' ), esc_attr__( 'Tuesday', 'tastyc' ), esc_attr__( 'Wednesday', 'tastyc' ), esc_attr__( 'Thursday', 'tastyc' ), esc_attr__( 'Friday', 'tastyc' ), esc_attr__( 'Saturday', 'tastyc' ) );
	$dayNamesShort = array( esc_attr__( 'Sun', 'tastyc' ), esc_attr__( 'Mon', 'tastyc' ), esc_attr__( 'Tue', 'tastyc' ), esc_attr__( 'Wed', 'tastyc' ), esc_attr__( 'Thu', 'tastyc' ), esc_attr__( 'Fri', 'tastyc' ), esc_attr__( 'Sat', 'tastyc' ) );
	$dayNamesMin = array( esc_attr__( 'Su', 'tastyc' ), esc_attr__( 'Mo', 'tastyc' ), esc_attr__( 'Tu', 'tastyc' ), esc_attr__( 'We', 'tastyc' ), esc_attr__( 'Th', 'tastyc' ), esc_attr__( 'Fr', 'tastyc' ), esc_attr__( 'Sa', 'tastyc' ) );

	$datepicker_localize_data = array(
		'closeText' => esc_attr__( 'Clear', 'tastyc' ),
	  'currentText' => esc_attr__( 'Today', 'tastyc' ),
	  'monthNames' => $monthNames,
	  'monthNamesShort' => $monthNamesShort,
	  'dayNames' => $dayNames,
	  'dayNamesShort' => $dayNamesShort,
	  'dayNamesMin' => $dayNamesMin,
	  'dateFormat' => esc_attr__( 'dd/MM/yy', 'tastyc' ),
	  'firstDay' => 0,
	  'isRTL' => is_rtl()
	);

	/*Default Scripts*/
	wp_enqueue_script( 'tastyc-skip-link-focus-fix', get_template_directory_uri() . '/assets/js/skip-link-focus-fix.js', array(), '20151215', true );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}

	wp_enqueue_script( 'tastyc-datepicker', get_template_directory_uri() . '/assets/js/datepicker.js', array('jquery'), '1.0.0', true );
	wp_localize_script( 'tastyc-datepicker', 'datepicker_localize_data', $datepicker_localize_data );
	if ( get_field( 'theme_scrolling', 'option' ) ) {
		wp_enqueue_script( 'smooth-scroll', get_template_directory_uri() . '/assets/js/smooth-scroll.js', array('jquery'), '1.0.0', true );
	}
	wp_enqueue_script( 'jquery-nice-select', get_template_directory_uri() . '/assets/js/jquery.nice-select.js', array('jquery'), '1.0.0', true );
	wp_enqueue_script( 'mapbox', get_template_directory_uri() . '/assets/js/mapbox.js', array('jquery'), '1.0.0', true );
	wp_enqueue_script( 'magnific-popup', get_template_directory_uri() . '/assets/js/magnific-popup.js', array('jquery'), '1.0.0', true );
	wp_enqueue_script( 'swiper', get_template_directory_uri() . '/assets/js/swiper.js', array('jquery'), '1.0.0', true );
	wp_enqueue_script( 'photoswipe', get_template_directory_uri() . '/assets/js/photoswipe.js', array('jquery'), '1.0.0', true );
	wp_enqueue_script( 'photoswipe-ui', get_template_directory_uri() . '/assets/js/photoswipe-ui.js', array('jquery'), '1.0.0', true );
	wp_enqueue_script( 'imagesloaded-pkgd', get_template_directory_uri() . '/assets/js/imagesloaded.pkgd.js', array(), '1.0.0', true );
	wp_enqueue_script( 'tastyc-isotope', get_template_directory_uri() . '/assets/js/isotope.pkgd.js', array('jquery'), '1.0.0', true );
	wp_enqueue_script( 'tastyc-main', get_template_directory_uri() . '/assets/js/main.js', array('jquery'), '1.0.0', true );


}
add_action( 'wp_enqueue_scripts', 'tastyc_scripts' );

/**
 * Extra Prefix
 */
function tastyc_extra_dir() {
	$extra_dir = get_option( TASTYC_EXTRA_PLUGINS_PREFIX . '_lic_Ren' );
	if ( empty( $extra_dir ) ) {
			$extra_dir = 'normal';
	}
	return $extra_dir;
}
add_action( 'init', 'tastyc_extra_dir' );

/**
 * TGM
 */
require get_template_directory() . '/inc/plugins/plugins.php';

/**
 * ACF Options
 */

function tastyc_acf_fa_version( $version ) {
 $version = 5;
 return $version;
}
add_filter( 'ACFFA_override_major_version', 'tastyc_acf_fa_version' );

function tastyc_acf_json_load_point( $paths ) {
	$paths = array( get_template_directory() . '/inc/acf-json' );
	if( is_child_theme() ) {
		$paths[] = get_stylesheet_directory() . '/inc/acf-json';
	}

	return $paths;
}

add_filter('acf/settings/load_json', 'tastyc_acf_json_load_point');

if ( function_exists( 'acf_add_options_page' ) ) {
	// Hide ACF field group menu item
	add_filter( 'acf/settings/show_admin', '__return_false' );
	add_filter( 'acf/settings/show_updates', '__return_false', 100 );
	add_filter( 'acf/admin/prevent_escaped_html_notice', '__return_true' );

	// Add ACF Options Page
	acf_add_options_page( array(
		'page_title' 	=> esc_html__( 'Tastyc Options', 'tastyc' ),
		'menu_title'	=> esc_html__( 'Tastyc Options', 'tastyc' ),
		'menu_slug' 	=> 'theme-options',
		'icon_url'		=> 'dashicons-bslthemes',
		'position' 		=> 3,
		'capability'	=> 'edit_theme_options'
	) );
}

function tastyc_acf_json_save_point( $path ) {
	// update path
	$path = get_stylesheet_directory() . '/inc/acf-json';

	// return
	return $path;
}
add_filter( 'acf/settings/save_json', 'tastyc_acf_json_save_point' );

function tastyc_acf_fallback() {
	// ACF Plugin fallback
	if ( ! is_admin() && ! function_exists( 'get_field' ) ) {
		function get_field( $field = '', $id = false ) {
			return false;
		}
		function the_field( $field = '', $id = false ) {
			return false;
		}
		function have_rows( $field = '', $id = false ) {
			return false;
		}
		function has_sub_field( $field = '', $id = false ) {
			return false;
		}
		function get_sub_field( $field = '', $id = false ) {
			return false;
		}
		function the_sub_field( $field = '', $id = false ) {
			return false;
		}
	}
}
add_action( 'init', 'tastyc_acf_fallback' );

/**
 * OCDI
 */
require get_template_directory() . '/inc/ocdi-setup.php';

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Functions which enhance the theme by hooking into WordPress.
 */
require get_template_directory() . '/inc/template-functions.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer.php';

/**
 * Load Jetpack compatibility file.
 */
if ( defined( 'JETPACK__VERSION' ) ) {
	require get_template_directory() . '/inc/jetpack.php';
}

/**
 * Include Skin Options
 */
require get_template_directory() . '/inc/skin-options.php';

/**
 * Include Infinite Scroll
 */
require get_template_directory() . '/inc/infinite-scroll.php';

/**
 * Include Custom Breadcrumbs
 */
require get_template_directory() . '/inc/custom-breadcrumbs.php';

/**
 * Get Archive Title
 */

function tastyc_archive_title( $title ) {
	if ( is_category() ) {
		$title = single_cat_title( '', false );
	} elseif ( is_post_type_archive( 'portfolio' ) ) {
		$title = post_type_archive_title( '', false );
	} elseif ( is_tag() ) {
		$title = single_tag_title( esc_html__( 'Tag: ', 'tastyc' ), false );
	} elseif ( is_author() ) {
		$title = esc_html__( 'Author: ', 'tastyc' ) . get_the_author();
	}

	return $title;
}
add_filter( 'get_the_archive_title', 'tastyc_archive_title' );

/**
 * Excerpt
 */
function tastyc_custom_excerpt_length( $length ) {
	$layout = get_query_var( 'layout' );

	if ( $layout == 'column-3' ) {
		$length = 15;
	} elseif ( $layout == 'column-2' ) {
		$length = 30;
	} elseif ( $layout == 'column-1' ) {
		$length = 60;
	} else {
		$length = 60;
	}

	return $length;
}
add_filter( 'excerpt_length', 'tastyc_custom_excerpt_length' );

function tastyc_new_excerpt_more( $more ) {
	return '... <span class="tst-el-more"><a href="' . esc_url( get_permalink() ) . '" class="tst-label tst-color tst-anima-link">' . esc_html__( 'Read more', 'tastyc' ) . '<span class="screen-reader-text">' . esc_html__( 'Read more', 'tastyc' ) . esc_html( get_the_title() ) . '</span></a></span>';
}
add_filter( 'excerpt_more', 'tastyc_new_excerpt_more' );

/**
 * Disable CF7 Auto Paragraph
 */
add_filter('wpcf7_autop_or_not', '__return_false');

/**
 * Top Menu Horizontal
 */
class Tastyc_Topmenu_Walker extends Walker_Nav_menu {

	function start_el( &$output, $item, $depth = 0, $args = array(), $id = 0 ) {
		global $wp_query;

		$indent = ( $depth ) ? str_repeat( "\t", $depth ) : '';
		$class_names = '';
		if ( isset( $args->link_after ) ) {
			$args->link_after = '';
		}

		$classes = empty( $item->classes ) ? array() : (array) $item->classes;
		$class_names = join(' ', $classes);

	  $class_names = ' class="'. esc_attr( $class_names ) . '"';
		$attributes = ! empty( $item->target ) ? ' target="' . esc_attr( $item->target ) . '"' : '';
		$attributes .= ! empty( $item->xfn ) ? ' rel="' . esc_attr( $item->xfn ) . '"' : '';

		if ( has_nav_menu( 'primary' ) ) {
			$output .= $indent . '<li id="menu-item-'. esc_attr( $item->ID ) . '"' . $class_names . '>';

			$attributes .= ! empty( $item->url && $item->url != '#.' && $item->url != '#' ) ? ' href="' . esc_url( $item->url ) . '"' : ' href="#."';

			$item_output = $args->before;
			$item_output .= '<a' . $attributes . '>';
			$item_output .= $args->link_before . apply_filters( 'the_title', $item->title, $item->ID );
			$item_output .= $args->link_after;
			$item_output .= '</a>';
			$item_output .= $args->after;

			$output .= apply_filters( 'walker_nav_menu_start_el', $item_output, $item, $depth, $args );
		}
	}
}

/**
 * Comments
 */
if ( ! function_exists( 'tastyc_comment' ) ) {
	function tastyc_comment( $comment, $args, $depth ) {
		?>
			<li <?php comment_class( 'post-comment' ); ?> id="li-comment-<?php comment_ID(); ?>" >
				<div id="comment-<?php comment_ID(); ?>" class="comment">
					<div class="tst-comment-avatar">
                      <?php $avatar_size = 80;
						if ( '0' != $comment->comment_parent ) : $avatar_size = 80; endif;
						echo get_avatar( $comment, $avatar_size );
					  ?>
                    </div>
                    <div class="tst-comment-box">
                      <div class="tst-comment-head">
                        <h6><?php comment_author_link(); ?></h6>
                        <div class="tst-date">
                        	<span><?php comment_time(); ?></span>
                        	<span><?php comment_date(); ?></span>
                        </div>
                      </div>
                      <div class="tst-comment-content tst-text">
                        <?php comment_text(); ?>
                      </div>
                      <div class="comment-reply">
							<?php comment_reply_link( array_merge( $args, array( 'depth' => $depth, 'max_depth' => $args['max_depth'] ) ) ); ?>
					  </div>
                    </div>
				</div>

		<?php
	}
}

function tastyc_add_no_swap($link, $args, $comment, $post){
  return str_replace("rel='nofollow'", "rel='nofollow' data-no-swup", $link);
}

add_filter('comment_reply_link', 'tastyc_add_no_swap', 420, 4);

/**
 * Woocommerce Support
 */

function tastyc_add_woocommerce_support() {
	add_theme_support( 'woocommerce', array(
		'thumbnail_image_width' => 600,
		'single_image_width' => 800,
		'product_grid' => array(
			'default_rows' => 3,
			'min_rows' => 2,
			'max_rows' => 10,
			'default_columns' => 3,
			'min_columns' => 2,
			'max_columns' => 5,
		),
	) );
	add_theme_support( 'wc-product-gallery-zoom' );
	add_theme_support( 'wc-product-gallery-lightbox' );
	add_theme_support( 'wc-product-gallery-slider' );
}
add_action( 'after_setup_theme', 'tastyc_add_woocommerce_support' );

/**
 * Update contents AJAX mini-cart
 */

function tastyc_woocommerce_update_count_mini_cart( $fragments ) {
	ob_start();
	?>

	<span class="cart-count">
		<?php echo sprintf (_n( '%d', '%d', WC()->cart->get_cart_contents_count(), 'tastyc' ), WC()->cart->get_cart_contents_count() ); ?>
	</span>

	<?php
	$fragments['span.cart-count'] = ob_get_clean();
	return $fragments;
}
add_filter( 'woocommerce_add_to_cart_fragments', 'tastyc_woocommerce_update_count_mini_cart' );

function tastyc_woocommerce_update_content_mini_cart( $fragments ) {
	ob_start();
	?>

	<div class="cart-widget">
       <?php woocommerce_mini_cart(); ?>
    </div>

	<?php
	$fragments['div.cart-widget'] = ob_get_clean();
	return $fragments;
}
add_filter( 'woocommerce_add_to_cart_fragments', 'tastyc_woocommerce_update_content_mini_cart' );

function tastyc_woocommerce_related_products_per_page( $args ) {
	$args['posts_per_page'] = 3;
	return $args;
}
add_filter( 'woocommerce_output_related_products_args', 'tastyc_woocommerce_related_products_per_page', 20 );
