<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Tastyc Team Widget.
 *
 * @since 1.0
 */

class Tastyc_Team_Widget extends Widget_Base {

	public function get_name() {
		return 'tastyc-team';
	}

	public function get_title() {
		return esc_html__( 'Team', 'tastyc-plugin' );
	}

	public function get_icon() {
		return 'eicon-parallax';
	}

	public function get_categories() {
		return [ 'tastyc-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'content_tab',
			[
				'label' => esc_html__( 'Content', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'image',
			[
				'label' => esc_html__( 'Image', 'tastyc-plugin' ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		$this->add_control(
			'name',
			[
				'label'       => esc_html__( 'Name', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter name', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Name', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'role',
			[
				'label'       => esc_html__( 'Role', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter role', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Role', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'link',
			[
				'label'       => esc_html__( 'Link', 'tastyc-plugin' ),
				'label_block' => true,
				'type' => Controls_Manager::URL,
				'show_external' => true,
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'social_tab',
			[
				'label' => esc_html__( 'Social Links', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'url', [
				'label'       => esc_html__( 'URL', 'oblo-plugin' ),
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'https://instagram.com/your_nickname/', 'oblo-plugin' ),
				'default' => '',
			]
		);

		$repeater->add_control(
			'icon', [
				'label'       => esc_html__( 'Icon', 'myour-plugin' ),
				'type'        => Controls_Manager::ICONS,
				'label_block' => true,
			]
		);

		$repeater->add_control(
			'title', [
				'label'       => esc_html__( 'Title', 'oblo-plugin' ),
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter title', 'oblo-plugin' ),
				'default' => esc_html__( 'Title', 'oblo-plugin' ),
			]
		);

		$this->add_control(
			'items',
			[
				'label' => esc_html__( 'Items', 'tastyc-plugin' ),
				'type' => Controls_Manager::REPEATER,
				'prevent_empty' => false,
				'fields' => $repeater->get_controls(),
				'title_field' => '{{{ title }}}',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'content_styling',
			[
				'label' => esc_html__( 'Content Styles', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'name_color',
			[
				'label' => esc_html__( 'Name Color', 'tastyc-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .tst-team-member h5' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'name_typography',
				'label' => esc_html__( 'Name Typography:', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-team-member h5',
			]
		);

		$this->add_control(
			'role_color',
			[
				'label' => esc_html__( 'Role Color', 'tastyc-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .tst-team-member .tst-text' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'role_typography',
				'label' => esc_html__( 'Role Typography:', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-team-member .tst-text',
			]
		);

		$this->add_control(
			'social_icon_color',
			[
				'label' => esc_html__( 'Icon Color', 'tastyc-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .tst-team-member .tst-member-social .tst-icon-link' => 'color: {{VALUE}};',
					'{{WRAPPER}} .tst-team-member .tst-member-social .tst-icon-link .e-font-icon-svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'social_icon_hover_color',
			[
				'label' => esc_html__( 'Icon Hover Color', 'tastyc-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .tst-team-member .tst-member-social .tst-icon-link:hover' => 'color: {{VALUE}};',
					'{{WRAPPER}} .tst-team-member .tst-member-social .tst-icon-link:hover .e-font-icon-svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$this->add_inline_editing_attributes( 'name', 'basic' );
		$this->add_inline_editing_attributes( 'role', 'basic' );

		?>

		<!-- team member -->
        <div class="tst-team-member">
          <?php if ( $settings['image'] ) : $image = wp_get_attachment_image_url( $settings['image']['id'], 'tastyc_900xAuto' ); ?>
          <a<?php if ( $settings['link'] ) : ?><?php if ( $settings['link']['is_external'] ) : ?> target="_blank"<?php endif; ?><?php if ( $settings['link']['nofollow'] ) : ?> rel="nofollow"<?php endif; ?> href="<?php echo esc_url( $settings['link']['url'] ); ?>"<?php endif; ?> class="tst-member-portrait-frame tst-anima-link">
            <img src="<?php echo esc_url( $image ); ?>" alt="<?php echo esc_attr( $settings['name'] ); ?>">
          </a>
          <?php endif; ?>
          <div class="tst-member-descr">
          	<?php if ( $settings['name']) : ?>
            <h5 class="tst-mb-15">
            	<a<?php if ( $settings['link'] ) : ?><?php if ( $settings['link']['is_external'] ) : ?> target="_blank"<?php endif; ?><?php if ( $settings['link']['nofollow'] ) : ?> rel="nofollow"<?php endif; ?> href="<?php echo esc_url( $settings['link']['url'] ); ?>"<?php endif; ?> class="tst-anima-link">
            		<span <?php echo $this->get_render_attribute_string( 'name' ); ?>>
						<?php echo wp_kses_post( $settings['name'] ); ?>
					</span>
            	</a>
            </h5>
            <?php endif; ?>
            <?php if ( $settings['role'] ) : ?>
            <div class="tst-text">
            	<span <?php echo $this->get_render_attribute_string( 'role' ); ?>>
					<?php echo wp_kses_post( $settings['role'] ); ?>
				</span>
            </div>
            <?php endif; ?>
            <div class="tst-spacer-sm"></div>
            <?php if ( $settings['items'] ) : ?>
            <div class="tst-member-social">
              <?php foreach ( $settings['items'] as $index => $item ) : ?>
              <a href="<?php echo esc_url( $item['url'] ); ?>" target="_blank" title="<?php echo esc_attr( $item['title'] ); ?>" class="tst-icon-link">
              	<?php \Elementor\Icons_Manager::render_icon( $item['icon'], [ 'aria-hidden' => 'true' ] ); ?>
              </a>
              <?php endforeach; ?>
            </div>
            <?php endif; ?>
          </div>
        </div>
        <!-- team member end -->

		<?php
	}

	/**
	 * Render widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function content_template() { ?>

		<#
		view.addInlineEditingAttributes( 'name', 'basic' );
		view.addInlineEditingAttributes( 'role', 'basic' );
		#>

		<!-- team member -->
        <div class="tst-team-member">
          <# if ( settings.image ) { #>
          <a<# if ( settings.link ) { #><# if ( settings.link.is_external ) { #> target="_blank"<# } #><# if ( settings.link.nofollow ) { #> rel="nofollow"<# } #> href="{{{ settings.link.url }}}"<# } #> class="tst-member-portrait-frame tst-anima-link">
            <img src="{{{ settings.image.url }}}" alt="{{{ settings.name }}}">
          </a>
          <# } #>
          <div class="tst-member-descr">
          	<# if ( settings.name ) { #>
            <h5 class="tst-mb-15">
            	<a<# if ( settings.link ) { #><# if ( settings.link.is_external ) { #> target="_blank"<# } #><# if ( settings.link.nofollow ) { #> rel="nofollow"<# } #> href="{{{ settings.link.url }}}"<# } #> class="tst-anima-link">
            		<span {{{ view.getRenderAttributeString( 'name' ) }}}>
        				{{{ settings.name }}}
        			</span>
            	</a>
            </h5>
            <# } #>
            <# if ( settings.role ) { #>
            <div class="tst-text">
            	<span {{{ view.getRenderAttributeString( 'role' ) }}}>
    				{{{ settings.role }}}
    			</span>
            </div>
            <# } #>
            <div class="tst-spacer-sm"></div>
            <# if ( settings.items ) { #>
            <div class="tst-member-social">
              <# _.each( settings.items, function( item, index ) {
			    var iconHTML = elementor.helpers.renderIcon( view, item.icon, { 'aria-hidden': true }, 'i' , 'object' );
			  #>
              <a href="{{{ item.url }}}" target="_blank" title="{{{ item.title }}}" class="tst-icon-link">
              	{{{ iconHTML.value }}}
              </a>
              <# }); #>
            </div>
            <# } #>
          </div>
        </div>
        <!-- team member end -->

	<?php }
}

Plugin::instance()->widgets_manager->register( new Tastyc_Team_Widget() );
