<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Tastyc Products Widget.
 *
 * @since 1.0
 */
class Tastyc_Products_Carousel_Widget extends Widget_Base {

	public function get_name() {
		return 'tastyc-products-carousel';
	}

	public function get_title() {
		return esc_html__( 'Products (Carousel)', 'tastyc-plugin' );
	}

	public function get_icon() {
		return 'eicon-parallax';
	}

	public function get_categories() {
		return [ 'tastyc-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'heading_tab',
			[
				'label' => esc_html__( 'Title', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'subtitle',
			[
				'label'       => esc_html__( 'Subtitle', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter subtitle', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Subtitle', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'title',
			[
				'label'       => esc_html__( 'Title', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter title', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Title', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'title_tag',
			[
				'label'       => esc_html__( 'Title Tag', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'h3',
				'options' => [
					'h1'  => __( 'H1', 'tastyc-plugin' ),
					'h2' => __( 'H2', 'tastyc-plugin' ),
					'h3' => __( 'H3', 'tastyc-plugin' ),
					'div' => __( 'DIV', 'tastyc-plugin' ),
				],
			]
		);

		$this->add_control(
			'description',
			[
				'label'       => esc_html__( 'Description', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter description', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Description', 'tastyc-plugin' ),
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'items_tab',
			[
				'label' => esc_html__( 'Items', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'source',
			[
				'label'       => esc_html__( 'Source', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'all',
				'options' => [
					'all'  => __( 'All', 'tastyc-plugin' ),
					'categories' => __( 'Taxonomy (Categories)', 'tastyc-plugin' ),
					'tags' => __( 'Taxonomy (Tags)', 'tastyc-plugin' ),
					'custom' => __( 'Custom', 'tastyc-plugin' ),
				],
			]
		);

		$this->add_control(
			'source_categories',
			[
				'label'       => esc_html__( 'Categories', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT2,
				'label_block' => true,
				'multiple' => true,
				'options' => $this->get_products_categories(),
				'condition' => [
		            'source' => 'categories'
		        ],
			]
		);

		$this->add_control(
			'source_tags',
			[
				'label'       => esc_html__( 'Tags', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT2,
				'label_block' => true,
				'multiple' => true,
				'options' => $this->get_products_tags(),
				'condition' => [
		            'source' => 'tags'
		        ],
			]
		);

		$this->add_control(
			'source_custom',
			[
				'label'       => esc_html__( 'Custom Products', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT2,
				'label_block' => true,
				'multiple' => true,
				'options' => $this->get_products_custom(),
				'condition' => [
		            'source' => 'custom'
		        ],
			]
		);

		$this->add_control(
			'limit',
			[
				'label'       => esc_html__( 'Number of Items', 'tastyc-plugin' ),
				'type'        => Controls_Manager::NUMBER,
				'placeholder' => 4,
				'default'     => 4,
			]
		);

		$this->add_control(
			'sort',
			[
				'label'       => esc_html__( 'Sorting By', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'menu_order',
				'options' => [
					'date'  => __( 'Date', 'tastyc-plugin' ),
					'title' => __( 'Title', 'tastyc-plugin' ),
					'rand' => __( 'Random', 'tastyc-plugin' ),
					'menu_order' => __( 'Order', 'tastyc-plugin' ),
				],
			]
		);

		$this->add_control(
			'order',
			[
				'label'       => esc_html__( 'Order', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'asc',
				'options' => [
					'asc'  => __( 'ASC', 'tastyc-plugin' ),
					'desc' => __( 'DESC', 'tastyc-plugin' ),
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'buttons_tab',
			[
				'label' => esc_html__( 'Buttons', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'more_btn_txt',
			[
				'label'       => esc_html__( 'Button (title)', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter button', 'tastyc-plugin' ),
				'default'     => esc_html__( 'All Products', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'more_btn_link',
			[
				'label'       => esc_html__( 'Button (link)', 'tastyc-plugin' ),
				'type'        => Controls_Manager::URL,
				'show_external' => true,
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'title_styling',
			[
				'label'     => esc_html__( 'Title', 'tastyc-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'subtitle_color',
			[
				'label'     => esc_html__( 'Subtitle Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .row .tst-suptitle' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'subtitle_typography',
				'label'     => esc_html__( 'Subtitle Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .row .tst-suptitle',
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__( 'Title Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .row .tst-title--h' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'label'     => esc_html__( 'Title Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .row .tst-title--h',
			]
		);

		$this->add_control(
			'desc_color',
			[
				'label'     => esc_html__( 'Description Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .row .tst-text' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'desc_typography',
				'label'     => esc_html__( 'Description Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .row .tst-text',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'items_styling',
			[
				'label'     => esc_html__( 'Items', 'tastyc-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

    $this->add_control(
			'item_bg_color',
			[
				'label'     => esc_html__( 'Item BG Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-menu-book-item' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'item_title_color',
			[
				'label'     => esc_html__( 'Title Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-menu-book-item h5 a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_title_typography',
				'label'     => esc_html__( 'Title Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-menu-book-item h5 a',
			]
		);

		$this->add_control(
			'item_desc_color',
			[
				'label'     => esc_html__( 'Description Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-menu-book-item .tst-text' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_desc_typography',
				'label'     => esc_html__( 'Description Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-menu-book-item .tst-text',
			]
		);

		$this->add_control(
			'item_price_color',
			[
				'label'     => esc_html__( 'Price Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-menu-book-item .tst-price' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_price_typography',
				'label'     => esc_html__( 'Price Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-menu-book-item .tst-price',
			]
		);

		$this->add_control(
			'item_btn_color',
			[
				'label'     => esc_html__( 'Button Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-menu-book-item .tst-btn svg path' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'item_btn_bgcolor',
			[
				'label'     => esc_html__( 'Button BG Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-menu-book-item .tst-btn' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_btn_typography',
				'label'     => esc_html__( 'Button Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-menu-book-item .tst-btn',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'buttons_styling',
			[
				'label'     => esc_html__( 'Buttons', 'tastyc-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'button_more_color',
			[
				'label'     => esc_html__( 'Button More Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-slider-navigation .tst-btn svg path' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'button_more_bgcolor',
			[
				'label'     => esc_html__( 'Button More BG Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-slider-navigation .tst-btn' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'button_more_typography',
				'label'     => esc_html__( 'Button More Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-slider-navigation .tst-btn',
			]
		);

		$this->add_control(
			'button_nav_color',
			[
				'label'     => esc_html__( 'Button Nav Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-slider-navigation .tst-nav .tst-slider-btn i' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'button_nav_bgcolor',
			[
				'label'     => esc_html__( 'Button Nav BG Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-slider-navigation .tst-nav .tst-slider-btn' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'button_nav_label_color',
			[
				'label'     => esc_html__( 'Button Nav Label Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-slider-navigation .tst-label' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'button_nav_label_typography',
				'label'     => esc_html__( 'Button Nav Label Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-slider-navigation .tst-label',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render Categories List.
	 *
	 * @since 1.0
	 */
	protected function get_products_categories() {
		$categories = [];

		$args = array(
			'type'			=> 'product',
			'child_of'		=> 0,
			'parent'		=> '',
			'orderby'		=> 'name',
			'order'			=> 'DESC',
			'hide_empty'	=> 1,
			'hierarchical'	=> 1,
			'taxonomy'		=> 'product_cat',
			'pad_counts'	=> false
		);

		$products_categories = get_categories( $args );

		foreach ( $products_categories as $category ) {
			$categories[$category->term_id] = $category->name;
		}

		return $categories;
	}

	protected function get_products_tags() {
		$tags = [];

		$args = array(
			'type'			=> 'product',
			'child_of'		=> 0,
			'parent'		=> '',
			'orderby'		=> 'name',
			'order'			=> 'DESC',
			'hide_empty'	=> 1,
			'hierarchical'	=> 1,
			'taxonomy'		=> 'product_tag',
			'pad_counts'	=> false
		);

		$products_tags = get_categories( $args );

		foreach ( $products_tags as $tag ) {
			$tags[$tag->term_id] = $tag->name;
		}

		return $tags;
	}

	protected function get_products_custom() {
		$items = [];

		$args = array(
			'post_type'			=> 'product',
			'post_status'		=> 'publish',
			'posts_per_page'	=> -1
		);

		$products_items = new \WP_Query( $args );

		while ( $products_items->have_posts() ) : $products_items->the_post();
			$items[get_the_ID()] = get_the_title();
		endwhile; wp_reset_postdata();

		return $items;
	}

	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$this->add_inline_editing_attributes( 'title', 'basic' );
		$this->add_inline_editing_attributes( 'subtitle', 'basic' );
		$this->add_inline_editing_attributes( 'description', 'basic' );
		$this->add_inline_editing_attributes( 'more_btn_txt', 'none' );

		if ( $settings['source'] == 'categories' ) {
			$ids = $settings['source_categories'];
		} elseif ( $settings['source'] == 'tags' ) {
			$ids = $settings['source_tags'];
		} elseif ( $settings['source'] == 'custom' ) {
			$ids = $settings['source_custom'];
		} else {
			$ids = '';
		}

		$args = array(
			'post_type'			=> 'product',
			'post_status'		=> 'publish',
			'orderby'			=> $settings['sort'],
			'order'				=> $settings['order'],
			'posts_per_page'	=> $settings['limit']
		);

		if ( $settings['source'] == 'categories' ) {
			$tax_array = array(
				array(
					'taxonomy' => 'product_cat',
					'field'    => 'id',
					'terms'    => $ids
				)
			);

			$args += array('tax_query' => $tax_array);
		}
		if ( $settings['source'] == 'tags' ) {
			$tax_array = array(
				array(
					'taxonomy' => 'product_tag',
					'field'    => 'id',
					'terms'    => $ids
				)
			);

			$args += array('tax_query' => $tax_array);
		}
		if ( $settings['source'] == 'custom' ) {
			$args += array( 'post__in' => $ids );
		}

		$products_loop = new \WP_Query( $args );

		?>

		<!-- container -->
		<div class="container">

            <div class="row align-items-center">
              <?php if ( $settings['title'] || $settings['subtitle'] || $settings['description'] ) : ?>
              <div class="col-lg-12">

                <!-- title -->
                <div class="text-center">
                  <?php if ( $settings['subtitle'] ) : ?>
                  <div class="tst-suptitle tst-suptitle-center tst-mb-15">
                  	<span <?php echo $this->get_render_attribute_string( 'subtitle' ); ?>><?php echo wp_kses_post( $settings['subtitle'] ); ?></span>
                  </div>
                  <?php endif; ?>
                  <?php if ( $settings['title'] ) : ?>
                  <<?php echo esc_attr( $settings['title_tag'] ); ?> class="tst-title--h tst-mb-30">
                  	<span <?php echo $this->get_render_attribute_string( 'title' ); ?>><?php echo wp_kses_post( $settings['title'] ); ?></span>
                  </<?php echo esc_attr( $settings['title_tag'] ); ?>>
                  <?php endif; ?>
                  <?php if ( $settings['description'] ) : ?>
                  <p class="tst-text">
                  	<span <?php echo $this->get_render_attribute_string( 'description' ); ?>><?php echo wp_kses_post( $settings['description'] ); ?></span>
                  </p>
                  <?php endif; ?>
                </div>
                <!-- title end-->

              </div>
          	  <?php endif; ?>

          	  <?php if ( class_exists( 'WooCommerce' ) ) : ?>
          	  <?php if ( $products_loop->have_posts() ) : ?>
              <div class="col-lg-12">

                <!-- slider -->
                <div class="swiper-container tst-slider">

                  <div class="swiper-wrapper">
                  	<?php while ( $products_loop->have_posts() ) : $products_loop->the_post();
                      	$product_id = get_the_ID();
                      	global $product;
						$product = wc_get_product( $product_id );
						$image = get_the_post_thumbnail_url( $product_id, 'tastyc_700x456' );
						$title = get_the_title( $product_id );
						$text = get_the_excerpt( $product_id );
						$href = get_the_permalink( $product_id );
					?>
                    <div class="swiper-slide">

                      <div class="tst-menu-book-item">
                        <?php if ( $image ) : ?>
                        <a href="<?php echo esc_url( $href ); ?>" class="tst-item-cover-frame tst-cursor-zoom">
                          <img src="<?php echo esc_url( $image ); ?>" alt="<?php echo esc_attr( $title ); ?>">
                          <span class="tst-overlay"></span>
                        </a>
                        <?php endif; ?>
                        <div class="tst-menu-book-descr">
                          <div class="tst-menu-book-name">
                            <h5 class="tst-mb-15">
                            	<a href="<?php echo esc_url( $href ); ?>">
                            		<?php echo wp_kses_post( $title ); ?>
                            	</a>
                            </h5>
                            <div class="tst-text"><?php echo wp_kses_post( $text ); ?></div>
                            <div class="tst-spacer-sm"></div>
                          </div>
                          <div class="tst-menu-book-bottom">
                            <div class="tst-menu-book-price">
                              <?php if( $product->is_on_sale() ) : ?>
                              <del>
                              	<span class="tst-price tst-old-price">
                              		<bdi>
                              			<?php echo wc_price( $product->get_regular_price() ); ?>
                              		</bdi>
                              	</span>
                              </del>
                          	  <?php endif; ?>
                              <div class="tst-price">
                              	<?php echo wc_price( $product->get_price() ); ?>
                              </div>
                            </div>
                            <a href="?add-to-cart=<?php echo esc_attr( $product_id ); ?>" class="tst-btn tst-cart-btn product_type_simple ajax_add_to_cart add_to_cart_button" data-product_id="<?php echo esc_attr( $product_id ); ?>" title="<?php echo esc_attr__( 'Add to Cart', 'tastyc-plugin' ); ?>">
                              <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 128 128">
                                <path
                                  d="M87.7,33.1l-0.8-10.8C86,10.4,76,1,64,1s-22.1,9.4-22.9,21.3l-0.8,10.8H28.8c-4.7,0-8.6,3.7-9,8.4l-5.4,75.9c0,0,0,0,0,0 c-0.2,2.5,0.7,5,2.4,6.8s4.1,2.9,6.6,2.9h81.3c2.5,0,4.9-1,6.6-2.9c1.7-1.8,2.6-4.3,2.4-6.8l-5.4-75.2c-0.4-5.1-4.6-9-9.7-9H87.7z M47.1,22.7C47.7,13.9,55.1,7,64,7s16.3,6.9,16.9,15.7l0.7,10.4H46.3L47.1,22.7z M102.3,42.6l5.4,75.2c0.1,0.8-0.2,1.6-0.8,2.3 c-0.6,0.6-1.4,1-2.2,1H23.4c-0.8,0-1.6-0.3-2.2-1s-0.9-1.4-0.8-2.3h0l5.4-75.9c0.1-1.6,1.4-2.8,3-2.8h11.1l-0.6,8 c-0.1,1.7,1.1,3.1,2.8,3.2c0.1,0,0.1,0,0.2,0c1.6,0,2.9-1.2,3-2.8l0.6-8.4h36.2l0.6,8.4c0.1,1.7,1.5,2.9,3.2,2.8 c1.7-0.1,2.9-1.5,2.8-3.2l-0.6-8h10.5C100.5,39.1,102.1,40.6,102.3,42.6z" />
                              </svg>
                            </a>
                          </div>
                        </div>
                      </div>

                    </div>
                    <?php endwhile; ?>

                  </div>

                </div>
                <!-- slider -->

              </div>
              <?php endif; wp_reset_postdata(); ?>
			  <?php endif; ?>

              <div class="col-lg-12">

                <!-- slider navigation -->
                <div class="tst-slider-navigation">
                  <?php if ( $settings['more_btn_txt'] ) : ?>
                  <a<?php if ( $settings['more_btn_link']['url'] ) : ?><?php if ( $settings['more_btn_link']['is_external'] ) : ?> target="_blank"<?php endif; ?><?php if ( $settings['more_btn_link']['nofollow'] ) : ?> rel="nofollow"<?php endif; ?> href="<?php echo esc_url( $settings['more_btn_link']['url'] ); ?>"<?php endif; ?> class="tst-btn">
                  	<span <?php echo $this->get_render_attribute_string( 'more_btn_txt' ); ?>>
                  		<?php echo wp_kses_post( $settings['more_btn_txt'] ); ?>
                  	</span>
                  </a>
              	  <?php endif; ?>
                  <div class="tst-slider-pagination tst-testi-pagination"></div>
                  <div class="tst-nav tst-right">
                    <div class="tst-label"><?php echo esc_html( 'Slider navigation', 'tastyc-plugin' ); ?></div>
                    <div class="tst-slider-btn tst-prev"><i class="fas fa-arrow-left"></i></div>
                    <div class="tst-slider-btn tst-next"><i class="fas fa-arrow-right"></i></div>
                  </div>
                </div>
                <!-- slider navigation end -->

              </div>

            </div>

        </div>

        <!-- container end -->

		<?php
	}
}

Plugin::instance()->widgets_manager->register( new Tastyc_Products_Carousel_Widget() );
