<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Tastyc Intro Map Widget.
 *
 * @since 1.0
 */
class Tastyc_Intro_Map_Widget extends Widget_Base {

	public function get_name() {
		return 'tastyc-intro-map';
	}

	public function get_title() {
		return esc_html__( 'Intro (Map)', 'tastyc-plugin' );
	}

	public function get_icon() {
		return 'eicon-parallax';
	}

	public function get_categories() {
		return [ 'tastyc-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'title_tab',
			[
				'label' => esc_html__( 'Title', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'title_value',
			[
				'label'       => esc_html__( 'Title Value', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'dynamic',
				'options' => [
					'dynamic'  => __( 'Dynamic', 'tastyc-plugin' ),
					'static' => __( 'Static', 'tastyc-plugin' ),
				],
			]
		);

		$this->add_control(
			'subtitle_value',
			[
				'label'       => esc_html__( 'Subtitle Value', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'dynamic',
				'options' => [
					'dynamic'  => __( 'Dynamic', 'tastyc-plugin' ),
					'static' => __( 'Static', 'tastyc-plugin' ),
				],
			]
		);

		$this->add_control(
			'description_value',
			[
				'label'       => esc_html__( 'Description Value', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'dynamic',
				'options' => [
					'dynamic'  => __( 'Dynamic', 'tastyc-plugin' ),
					'static' => __( 'Static', 'tastyc-plugin' ),
				],
			]
		);

		$this->add_control(
			'subtitle',
			[
				'label'       => esc_html__( 'Subtitle', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter subtitle', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Subtitle', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'title',
			[
				'label'       => esc_html__( 'Title', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter title', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Title', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'title_tag',
			[
				'label'       => esc_html__( 'Title Tag', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'h1',
				'options' => [
					'h1'  => __( 'H1', 'tastyc-plugin' ),
					'h2' => __( 'H2', 'tastyc-plugin' ),
					'h3' => __( 'H3', 'tastyc-plugin' ),
					'div' => __( 'DIV', 'tastyc-plugin' ),
				],
			]
		);

		$this->add_control(
			'description',
			[
				'label'       => esc_html__( 'Description', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter description', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Description', 'tastyc-plugin' ),
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'breadcrumbs_tab',
			[
				'label' => esc_html__( 'Breadcrumbs', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'breadcrumbs',
			[
				'label'       => esc_html__( 'Breadcrumbs', 'tastyc-plugin' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => __( 'Yes', 'tastyc-plugin' ),
				'label_off' => __( 'No', 'tastyc-plugin' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'map_tab',
			[
				'label' => esc_html__( 'Map', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'map_long',
			[
				'label'       => esc_html__( 'Long', 'tastyc-plugin' ),
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter long value', 'tastyc-plugin' ),
				'default'     => esc_html__( '70.123', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'map_lat',
			[
				'label'       => esc_html__( 'Lat', 'tastyc-plugin' ),
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter lat value', 'tastyc-plugin' ),
				'default'     => esc_html__( '50.123', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'map_zoom',
			[
				'label'       => esc_html__( 'Zoom', 'tastyc-plugin' ),
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter zoom value', 'tastyc-plugin' ),
				'default'     => esc_html__( '13', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'map_style',
			[
				'label'    => esc_html__( 'Map Style', 'tastyc-plugin' ),
				'type'  => Controls_Manager::SELECT,
				'default' => '0',
				'options' => [
					'0' => __( 'Dark', 'tastyc-plugin' ),
					'1'  => __( 'Light', 'tastyc-plugin' ),
					'2'  => __( 'Default', 'tastyc-plugin' ),
				],
			]
		);
		 

		$this->end_controls_section();

		$this->start_controls_section(
			'title_styling',
			[
				'label'     => esc_html__( 'Title', 'tastyc-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'subtitle_color',
			[
				'label'     => esc_html__( 'Subtitle Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-banner .tst-suptitle' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'subtitle_typography',
				'label'     => esc_html__( 'Subtitle Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-banner .tst-suptitle',
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__( 'Title Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-banner .tst-title--h' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'label'     => esc_html__( 'Title Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-banner .tst-title--h',
			]
		);

		$this->add_control(
			'desc_color',
			[
				'label'     => esc_html__( 'Description Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-banner .tst-text' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'desc_typography',
				'label'     => esc_html__( 'Description Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-banner .tst-text',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'breadcrumbs_styling',
			[
				'label'     => esc_html__( 'Breadcrumbs', 'tastyc-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'breadcrumbs_bgcolor',
			[
				'label'     => esc_html__( 'Background Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-breadcrumbs' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'breadcrumbs_link_color',
			[
				'label'     => esc_html__( 'Background Link Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-breadcrumbs li a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'breadcrumbs_link_typography',
				'label'     => esc_html__( 'Background Link Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-breadcrumbs li a',
			]
		);

		$this->end_controls_section();
	}


	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$this->add_inline_editing_attributes( 'subtitle', 'basic' );
		$this->add_inline_editing_attributes( 'title', 'basic' );
		$this->add_inline_editing_attributes( 'description', 'basic' );

		$page_id = get_queried_object_id();

		if ( $page_id ) {
			$intro_title = get_field( 'intro_title', $page_id );
			if ( !$intro_title ) {
				$intro_title = get_the_title( $page_id );
			}

			$intro_sub = get_field( 'intro_subtitle', $page_id );
			$intro_text = get_field( 'intro_description', $page_id );

			if ( $intro_title && $settings['title_value'] == 'dynamic'  ) {
				$settings['title'] = $intro_title;
			}
			if ( $intro_sub && $settings['subtitle_value'] == 'dynamic' ) {
				$settings['subtitle'] = $intro_sub;
			}
			if ( $intro_text && $settings['description_value'] == 'dynamic' ) {
				$settings['description'] = $intro_text;
			}
		}

		$map_key = get_field( 'mapbox_key', 'options' );

		?>

		<!-- banner -->
		<div class="tst-banner tst-small-banner">
			<div class="tst-cover-frame">
	          <div class="tst-map-frame tst-parallax">
	            <div id="map" data-key="<?php echo esc_attr( $map_key ); ?>" data-long="<?php echo esc_attr( $settings['map_long'] ); ?>" data-lat="<?php echo esc_attr( $settings['map_lat'] ); ?>" data-zoom="<?php echo esc_attr( $settings['map_zoom'] ); ?>" data-style="<?php echo esc_attr( $settings['map_style'] ); ?>" class="tst-map"></div>
	          </div>
	          <div class="tst-overlay tst-with-map"></div>
	          <div class="tst-lock"><i class="fas fa-lock"></i></div>
	        </div>
			<div class="tst-banner-content-frame tst-with-map">
			  <div class="container">
			  	<div class="row">
              		<div class="col-lg-6">
					    <div class="tst-main-title-frame">
					      <div class="tst-main-title">
					      	<?php if ( $settings['subtitle'] ) : ?>
					        <div class="tst-suptitle tst-suptitle-mobile-center tst-text-shadow tst-white-2 tst-mb-15">
					        	<span <?php echo $this->get_render_attribute_string( 'subtitle' ); ?>>
						          	<?php echo wp_kses_post( $settings['subtitle'] ); ?>
						        </span>
					        </div>
					        <?php endif; ?>
					        <?php if ( $settings['title'] ) : ?>
					        <<?php echo esc_attr( $settings['title_tag'] ); ?> class="tst-title--h tst-white-2 tst-text-shadow tst-mb-30">
					        	<span <?php echo $this->get_render_attribute_string( 'title' ); ?>>
						          	<?php echo wp_kses_post( $settings['title'] ); ?>
						        </span>
					    	</<?php echo esc_attr( $settings['title_tag'] ); ?>>
					        <?php endif; ?>
					        <?php if ( $settings['description'] ) : ?>
					        <div class="tst-text tst-text-shadow tst-text-lg tst-white-2 tst-mb-30">
					        	<span <?php echo $this->get_render_attribute_string( 'description' ); ?>>
						          	<?php echo wp_kses_post( $settings['description'] ); ?>
						        </span>
					        </div>
					        <?php endif; ?>
					        <?php if ( $settings['breadcrumbs'] == 'yes' ) : tastyc_breadcrumbs( $page_id ); endif; ?>
					      </div>
					    </div>
					</div>
				</div>
			  </div>
			</div>
		</div>
		<!-- banner end -->

		<?php
	}

	/**
	 * Render widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function content_template() {
		?>

		<#
		view.addInlineEditingAttributes( 'subtitle', 'basic' );
		view.addInlineEditingAttributes( 'title', 'basic' );
		view.addInlineEditingAttributes( 'description', 'basic' );
		#>

		<!-- banner -->
		<div class="tst-banner tst-small-banner">
			<div class="tst-cover-frame">
	          <div class="tst-map-frame tst-parallax">
	            <div id="map" data-long="{{{ settings.map_long }}}" data-lat="{{{ settings.map_lat }}}" data-zoom="{{{ settings.map_zoom }}}" data-style="{{{ settings.map_style }}}" class="tst-map"></div>
	          </div>
	          <div class="tst-overlay tst-with-map"></div>
	          <div class="tst-lock"><i class="fas fa-lock"></i></div>
	        </div>
			<div class="tst-banner-content-frame tst-with-map">
			  <div class="container">
			  	<div class="row">
              		<div class="col-lg-6">
					    <div class="tst-main-title-frame">
					      <div class="tst-main-title">
					      	<# if ( settings.subtitle ) { #>
					        <div class="tst-suptitle tst-suptitle-mobile-center tst-text-shadow tst-white-2 tst-mb-15">
					        	<span {{{ view.getRenderAttributeString( 'subtitle' ) }}}>
						          	{{{ settings.subtitle }}}
						        </span>
					        </div>
					        <# } #>
					        <# if ( settings.title ) { #>
					        <{{{ settings.title_tag }}} class="tst-title--h tst-white-2 tst-text-shadow tst-mb-30">
					        	<span {{{ view.getRenderAttributeString( 'title' ) }}}>
						          	{{{ settings.title }}}
						        </span>
					    	</{{{ settings.title_tag }}}>
					        <# } #>
					        <# if ( settings.description ) { #>
					        <div class="tst-text tst-text-shadow tst-text-lg tst-white-2 tst-mb-30">
					        	<span {{{ view.getRenderAttributeString( 'description' ) }}}>
						          	{{{ settings.description }}}
						        </span>
					        </div>
					        <# } #>
					      </div>
					    </div>
					</div>
				</div>
			  </div>
			</div>
		</div>
		<!-- banner end -->

		<?php
	}
}

Plugin::instance()->widgets_manager->register( new Tastyc_Intro_Map_Widget() );
