<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Tastyc Gallery Widget.
 *
 * @since 1.0
 */
class Tastyc_Gallery_Widget extends Widget_Base {

	public function get_name() {
		return 'tastyc-gallery';
	}

	public function get_title() {
		return esc_html__( 'Gallery', 'tastyc-plugin' );
	}

	public function get_icon() {
		return 'eicon-parallax';
	}

	public function get_categories() {
		return [ 'tastyc-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'filters_tab',
			[
				'label' => esc_html__( 'Filters', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'filters_note',
			[
				'type' => \Elementor\Controls_Manager::RAW_HTML,
				'raw' => esc_html__( 'Filters show only with pagination "Infinite Scrolling" or "No"', 'tastyc-plugin' ),
				'content_classes' => 'elementor-descriptor',
			]
		);

		$this->add_control(
			'filters',
			[
				'label' => esc_html__( 'Show Filters', 'tastyc-plugin' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => __( 'Show', 'tastyc-plugin' ),
				'label_off' => __( 'Hide', 'tastyc-plugin' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'items_tab',
			[
				'label' => esc_html__( 'Items', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'source',
			[
				'label'       => esc_html__( 'Source', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'all',
				'options' => [
					'all'  => __( 'All', 'tastyc-plugin' ),
					'categories' => __( 'Categories', 'tastyc-plugin' ),
				],
			]
		);

		$this->add_control(
			'source_categories',
			[
				'label'       => esc_html__( 'Source', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT2,
				'label_block' => true,
				'multiple' => true,
				'options' => $this->get_gallery_categories(),
				'condition' => [
		            'source' => 'categories'
		        ],
			]
		);

		$this->add_control(
			'limit',
			[
				'label'       => esc_html__( 'Number of Items', 'tastyc-plugin' ),
				'type'        => Controls_Manager::NUMBER,
				'placeholder' => 8,
				'default'     => 8,
			]
		);

		$this->add_control(
			'sort',
			[
				'label'       => esc_html__( 'Sorting By', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'menu_order',
				'options' => [
					'date'  => __( 'Date', 'tastyc-plugin' ),
					'title' => __( 'Title', 'tastyc-plugin' ),
					'rand' => __( 'Random', 'tastyc-plugin' ),
					'menu_order' => __( 'Order', 'tastyc-plugin' ),
				],
			]
		);

		$this->add_control(
			'order',
			[
				'label'       => esc_html__( 'Order', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'asc',
				'options' => [
					'asc'  => __( 'ASC', 'tastyc-plugin' ),
					'desc' => __( 'DESC', 'tastyc-plugin' ),
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'pagination_tab',
			[
				'label' => esc_html__( 'Pagination', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'pagination',
			[
				'label'       => esc_html__( 'Pagination Type', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'no',
				'options' => [
					'pages' => __( 'Pages', 'tastyc-plugin' ),
					'button' => __( 'Button', 'tastyc-plugin' ),
					'scroll' => __( 'Infinite Scrolling', 'tastyc-plugin' ),
					'no' => __( 'No', 'tastyc-plugin' ),
				],
			]
		);

		$this->add_control(
			'more_btn_txt',
			[
				'label'       => esc_html__( 'Button (title)', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter button', 'tastyc-plugin' ),
				'default'     => esc_html__( 'All Posts', 'tastyc-plugin' ),
				'condition' => [
		            'pagination' => 'button'
		        ],
			]
		);

		$this->add_control(
			'more_btn_link',
			[
				'label'       => esc_html__( 'Button (link)', 'tastyc-plugin' ),
				'type'        => Controls_Manager::URL,
				'show_external' => true,
				'condition' => [
		            'pagination' => 'button'
		        ],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'settings_tab',
			[
				'label' => esc_html__( 'Settings', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'image_title_show',
			[
				'label' => esc_html__( 'Show Image Title?', 'tastyc-plugin' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => __( 'Show', 'tastyc-plugin' ),
				'label_off' => __( 'Hide', 'tastyc-plugin' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->add_control(
			'image_category_show',
			[
				'label' => esc_html__( 'Show Image Category?', 'tastyc-plugin' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => __( 'Show', 'tastyc-plugin' ),
				'label_off' => __( 'Hide', 'tastyc-plugin' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'filters_styling',
			[
				'label'     => esc_html__( 'Filters', 'tastyc-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'filters_color',
			[
				'label'     => esc_html__( 'Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .filter__item a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'filters_active_color',
			[
				'label'     => esc_html__( 'Active Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .filter__item.active a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'filters_typography',
				'selector' => '{{WRAPPER}} .filter__item a',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'items_styling',
			[
				'label'     => esc_html__( 'Items', 'tastyc-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'item_title_color',
			[
				'label'     => esc_html__( 'Title Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .itemGallery__details .title--h4' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_title_typography',
				'label'     => esc_html__( 'Title Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .itemGallery__details .title--h4',
			]
		);

		$this->add_control(
			'item_subtitle_color',
			[
				'label'     => esc_html__( 'Subtitle Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .itemGallery__details .title--overhead' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_text_typography',
				'label'     => esc_html__( 'Subtitle Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .itemGallery__details .title--overhead',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render Categories List.
	 *
	 * @since 1.0
	 */
	protected function get_gallery_categories() {
		$categories = [];

		$args = array(
			'type'			=> 'post',
			'child_of'		=> 0,
			'parent'		=> '',
			'orderby'		=> 'name',
			'order'			=> 'DESC',
			'hide_empty'	=> 1,
			'hierarchical'	=> 1,
			'taxonomy'		=> 'gallery_categories',
			'pad_counts'	=> false
		);

		$gallery_categories = get_categories( $args );

		foreach ( $gallery_categories as $category ) {
			$categories[$category->term_id] = $category->name;
		}

		return $categories;
	}

	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$this->add_inline_editing_attributes( 'title', 'basic' );
		$this->add_inline_editing_attributes( 'description', 'advanced' );

		$paged = ( get_query_var( 'paged' ) ) ? absint( get_query_var( 'paged' ) ) : 1;
		$page_id = get_the_ID();

		if ( $settings['source'] == 'all' ) {
			$cat_ids = '';
		} else {
			$cat_ids = $settings['source_categories'];
		}

		$cat_args = array(
			'type'			=> 'post',
			'child_of'		=> 0,
			'parent'		=> '',
			'orderby'		=> 'name',
			'order'			=> 'DESC',
			'hide_empty'	=> 1,
			'hierarchical'	=> 1,
			'taxonomy'		=> 'gallery_categories',
			'pad_counts'	=> false,
			'include'		=> $cat_ids
		);

		$gl_categories = get_categories( $cat_args );

		$args = array(
			'post_type'			=> 'gallery',
			'post_status'		=> 'publish',
			'orderby'			=> $settings['sort'],
			'order'				=> $settings['order'],
			'posts_per_page'	=> $settings['limit'],
			'paged' 			=> $paged
		);

		if( $settings['source'] == 'categories' ) {
			$tax_array = array(
				array(
					'taxonomy' => 'gallery_categories',
					'field'    => 'id',
					'terms'    => $cat_ids
				)
			);

			$args += array('tax_query' => $tax_array);
		}

		$template = 'gallery';

		$q = new \WP_Query( $args );

		?>

		<div class="container">
		    <div class="row">
				<?php if ( $settings['filters'] && $gl_categories && $settings['pagination'] != 'pages' ) : ?>
				<div class="col-12">
				<!-- Filters -->
					<div class="filter gallery-filter">
			            <?php if ( $settings['filters'] && $gl_categories ) : ?>
			            <ul>
				            <li class="filter__item active" data-filter="*"><a class="filter__link active" href="#filter"><?php echo esc_html__( 'All', 'tastyc-plugin' ); ?></a></li>
				            <?php foreach ( $gl_categories as $category ) : ?>
				            <li class="filter__item" data-filter=".category-<?php echo esc_attr( $category->slug ); ?>"><a class="filter__link" href="#filter"><?php echo esc_html( $category->name ); ?></a></li>
				            <?php endforeach; ?>
			            </ul>
			            <?php endif; ?>
			        </div>
				</div>
				<?php endif; ?>
			</div>

			<?php if ( $q->have_posts() ) : ?>
		    <div class="flex-container js-gallery js-lightbox js-sort row">
				<?php while ( $q->have_posts() ) : $q->the_post();
					set_query_var( 'show_title', $settings['image_title_show'] );
					set_query_var( 'show_category', $settings['image_category_show'] );

					get_template_part( 'template-parts/content', 'gallery' );
				endwhile; ?>
			</div>

			<?php if ( $settings['pagination'] == 'pages' ) : ?>
			<div class="tst-a tst-pagination">
				<?php
					$big = 999999999; // need an unlikely integer

					echo paginate_links( array(
						'base' => str_replace( $big, '%#%', esc_url( get_pagenum_link( $big ) ) ),
						'format' => '?paged=%#%',
						'current' => max( 1, get_query_var('paged') ),
						'total' => $q->max_num_pages,
						'prev_text' => '<i class="icon-arrow-left"></i>',
						'next_text' => '<i class="icon-arrow-right"></i>',
					) );
				?>
			</div>
			<?php endif; ?>

			<?php if ( $settings['pagination'] == 'button' && $settings['more_btn_link'] ) : ?>
			<div class="bts text-center">
				<a class="tst-btn" href="<?php echo esc_url( $settings['more_btn_link']['url'] ); ?>"<?php if ( $settings['more_btn_link']['is_external'] ) : ?> target="_blank"<?php endif; ?><?php if ( $settings['more_btn_link']['nofollow'] ) : ?> rel="nofollow"<?php endif; ?>><?php echo esc_html( $settings['more_btn_txt'] ); ?></a>
			</div>
			<?php endif; ?>

			<?php
			if ( $settings['pagination'] == 'scroll' ) :
				$infinite_scrolling_data = array(
					'url'   => admin_url( 'admin-ajax.php' ),
					'max_num' => $q->max_num_pages,
					'page_id' => $page_id,
					'order_by' => $settings['sort'],
					'order' => $settings['order'],
					'per_page' => $settings['limit'],
					'source' => $settings['source'],
					'temp' => $template,
					'cat_ids' => $cat_ids,
					'title_show' => $settings['image_title_show'],
					'category_show' => $settings['image_category_show'],
				);

				wp_enqueue_script( 'tastyc-gallery-infinite-scroll-el', get_template_directory_uri() . '/assets/js/gallery-infinite-scroll-el.js', array( 'jquery' ), '1.0', true );
				wp_localize_script( 'tastyc-gallery-infinite-scroll-el', 'ajax_gallery_infinite_scroll_data', $infinite_scrolling_data );
			endif;
			?>

			<?php else :
				get_template_part( 'template-parts/content', 'none' );
			endif; wp_reset_postdata(); ?>

		</div>

		<?php
	}
}

Plugin::instance()->widgets_manager->register( new Tastyc_Gallery_Widget() );
