<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Tastyc Footer Info Widget.
 *
 * @since 1.0
 */
class Tastyc_Footer_Info_Widget extends Widget_Base {

	public function get_name() {
		return 'tastyc-footer-info';
	}

	public function get_title() {
		return esc_html__( 'Footer Info', 'tastyc-plugin' );
	}

	public function get_icon() {
		return 'eicon-parallax';
	}

	public function get_categories() {
		return [ 'tastyc-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'content_tab',
			[
				'label' => esc_html__( 'Content', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'title',
			[
				'label'       => esc_html__( 'Title', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter title', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Title', 'tastyc-plugin' ),
			]
		);

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'label', [
				'label'       => esc_html__( 'Label', 'tastyc-plugin' ),
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter label', 'tastyc-plugin' ),
				'default' => esc_html__( 'Enter label', 'tastyc-plugin' ),
			]
		);

		$repeater->add_control(
			'value', [
				'label'       => esc_html__( 'Value', 'tastyc-plugin' ),
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter value', 'tastyc-plugin' ),
				'default' => esc_html__( 'Enter value', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'items',
			[
				'label' => esc_html__( 'Items', 'tastyc-plugin' ),
				'type' => Controls_Manager::REPEATER,
				'prevent_empty' => false,
				'fields' => $repeater->get_controls(),
				'title_field' => '{{{ label }}}',
			]
		);

		$this->add_control(
			'link_text',
			[
				'label'       => esc_html__( 'Link Text', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter your link text', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Read More', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'link',
			[
				'label' => esc_html__( 'Link URL', 'tastyc-plugin' ),
				'label_block' => true,
				'type' => Controls_Manager::URL,
				'show_external' => false,
			]
		);

		$this->add_responsive_control(
			'align',
			[
				'label' => __( 'Alignment', 'tastyc-plugin' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => __( 'Left', 'tastyc-plugin' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'tastyc-plugin' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'tastyc-plugin' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}}' => 'text-align: {{VALUE}};',
				],
				'default'	=> 'left',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'content_styling',
			[
				'label' => esc_html__( 'Content Styles', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'title_color',
			[
				'label' => esc_html__( 'Title Color', 'tastyc-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .tst-mb-60 h5' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'title_typography',
				'label' => esc_html__( 'Title Typography:', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-mb-60 h5',
			]
		);

		$this->add_control(
			'item_label_color',
			[
				'label' => esc_html__( 'Item Label Color', 'tastyc-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .tst-footer-contact li .tst-label' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'item_label_typography',
				'label' => esc_html__( 'Item Label Typography:', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-footer-contact li .tst-label',
			]
		);

		$this->add_control(
			'item_value_color',
			[
				'label' => esc_html__( 'Item Value Color', 'tastyc-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .tst-footer-contact li .tst-text' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'item_value_typography',
				'label' => esc_html__( 'Item Value Typography:', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-footer-contact li .tst-text',
			]
		);

		$this->add_control(
			'link_color',
			[
				'label' => esc_html__( 'Link Color', 'tastyc-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .tst-mb-60 .tst-label' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'link_typography',
				'label' => esc_html__( 'Link Typography:', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-mb-60 .tst-label',
			]
		);

		$this->end_controls_section();

	}


	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$this->add_inline_editing_attributes( 'title', 'basic' );
		$this->add_inline_editing_attributes( 'link_text', 'none' );

		?>

		<div class="tst-mb-60">
			<?php if ( $settings['title'] ) : ?>
        	<h5 class="tst-mb-30">
        		<span <?php echo $this->get_render_attribute_string( 'title' ); ?>>
					<?php echo wp_kses_post( $settings['title'] ); ?>
				</span>
        	</h5>
        	<?php endif; ?>
        	<?php if ( $settings['items'] ) : ?>
        	<ul class="tst-footer-contact tst-mb-30">
            	<?php foreach ( $settings['items'] as $index => $item ) :
					$item_label = $this->get_repeater_setting_key( 'label', 'items', $index );
		  			$this->add_inline_editing_attributes( $item_label, 'basic' );

		  			$item_value = $this->get_repeater_setting_key( 'value', 'items', $index );
		  			$this->add_inline_editing_attributes( $item_value, 'basic' );
				?>
            	<li>
            		<span class="tst-label">
            			<span <?php echo $this->get_render_attribute_string( $item_label ); ?>>
							<?php echo wp_kses_post( $item['label'] ); ?>
						</span>
            		</span>
            		<span class="tst-text">
            			<span <?php echo $this->get_render_attribute_string( $item_value ); ?>>
							<?php echo wp_kses_post( $item['value'] ); ?>
						</span>
            		</span>
            	</li>
          		<?php endforeach; ?>
          	</ul>
        	<?php endif; ?>
        	<?php if ( $settings['link_text'] ) : ?>
        	<a<?php if ( $settings['link'] ) : ?><?php if ( $settings['link']['is_external'] ) : ?> target="_blank"<?php endif; ?><?php if ( $settings['link']['nofollow'] ) : ?> rel="nofollow"<?php endif; ?> href="<?php echo esc_url( $settings['link']['url'] ); ?>"<?php endif; ?> class="tst-label tst-color tst-anima-link">
        		<span <?php echo $this->get_render_attribute_string( 'link_text' ); ?>>
							<?php echo wp_kses_post( $settings['link_text'] ); ?>
						</span>
						<span class="screen-reader-text"><?php echo esc_html( $settings['link_text'] ) . esc_html( $settings['title'] ); ?></span>
        	</a>
        	<?php endif; ?>
        </div>

		<?php
	}

	/**
	 * Render widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function content_template() {
		?>
		<#
		view.addInlineEditingAttributes( 'title', 'basic' );
		view.addInlineEditingAttributes( 'link_text', 'none' );
		#>

		<div class="tst-mb-60">
			<# if ( settings.title ) { #>
        	<h5 class="tst-mb-30">
        		<span {{{ view.getRenderAttributeString( 'title' ) }}}>
	    			{{{ settings.title }}}
	    		</span>
        	</h5>
        	<# } #>
        	<# if ( settings.items ) { #>
        	<ul class="tst-footer-contact tst-mb-30">
            	<# _.each( settings.items, function( item, index ) {
				var item_label = view.getRepeaterSettingKey( 'label', 'items', index );
				view.addInlineEditingAttributes( item_label, 'basic' );

				var item_value = view.getRepeaterSettingKey( 'value', 'items', index );
				view.addInlineEditingAttributes( item_value, 'basic' );
				#>
            	<li>
            		<span class="tst-label">
            			<span {{{ view.getRenderAttributeString( item_label ) }}}>
							{{{ item.item_label }}}
						</span>
            		</span>
            		<span class="tst-text">
            			<span {{{ view.getRenderAttributeString( item_value ) }}}>
							{{{ item.item_value }}}
						</span>
            		</span>
            	</li>
          		<# }); #>
          	</ul>
        	<# } #>
        	<# if ( settings.link_text ) { #>
        	<a<# if ( settings.link ) { #><# if ( settings.link.is_external ) { #> target="_blank"<# } #><# if ( settings.link.nofollow ) { #> rel="nofollow"<# } #> href="{{{ settings.link.url }}}"<# } #> class="tst-label tst-color tst-anima-link">
        		<span {{{ view.getRenderAttributeString( 'link_text' ) }}}>
	    			{{{ settings.link_text }}}
	    		</span>
        	</a>
        	<# } #>
        </div>

		<?php
	}
}

Plugin::instance()->widgets_manager->register( new Tastyc_Footer_Info_Widget() );
