<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Tastyc Blog Grid Widget.
 *
 * @since 1.0
 */
class Tastyc_Blog_Grid_Widget extends Widget_Base {

	public function get_name() {
		return 'tastyc-blog-grid';
	}

	public function get_title() {
		return esc_html__( 'Blog (Grid)', 'tastyc-plugin' );
	}

	public function get_icon() {
		return 'eicon-parallax';
	}

	public function get_categories() {
		return [ 'tastyc-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'heading_tab',
			[
				'label' => esc_html__( 'Title', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'subtitle',
			[
				'label'       => esc_html__( 'Subtitle', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter subtitle', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Subtitle', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'title',
			[
				'label'       => esc_html__( 'Title', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter title', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Title', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'title_tag',
			[
				'label'       => esc_html__( 'Title Tag', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'h3',
				'options' => [
					'h1'  => __( 'H1', 'tastyc-plugin' ),
					'h2' => __( 'H2', 'tastyc-plugin' ),
					'h3' => __( 'H3', 'tastyc-plugin' ),
					'h4' => __( 'H4', 'tastyc-plugin' ),
					'div' => __( 'DIV', 'tastyc-plugin' ),
				],
			]
		);

		$this->add_control(
			'description',
			[
				'label'       => esc_html__( 'Description', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter description', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Description', 'tastyc-plugin' ),
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'items_tab',
			[
				'label' => esc_html__( 'Items', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'source',
			[
				'label'       => esc_html__( 'Source', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'all',
				'options' => [
					'all'  => __( 'All', 'tastyc-plugin' ),
					'categories' => __( 'Categories', 'tastyc-plugin' ),
				],
			]
		);

		$this->add_control(
			'source_categories',
			[
				'label'       => esc_html__( 'Source', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT2,
				'label_block' => true,
				'multiple' => true,
				'options' => $this->get_blog_categories(),
				'condition' => [
		            'source' => 'categories'
		        ],
			]
		);

		$this->add_control(
			'limit',
			[
				'label'       => esc_html__( 'Number of Items', 'tastyc-plugin' ),
				'type'        => Controls_Manager::NUMBER,
				'placeholder' => 6,
				'default'     => 6,
			]
		);

		$this->add_control(
			'sort',
			[
				'label'       => esc_html__( 'Sorting By', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'date',
				'options' => [
					'date'  => __( 'Date', 'tastyc-plugin' ),
					'title' => __( 'Title', 'tastyc-plugin' ),
					'rand' => __( 'Random', 'tastyc-plugin' ),
					'menu_order' => __( 'Order', 'tastyc-plugin' ),
				],
			]
		);

		$this->add_control(
			'order',
			[
				'label'       => esc_html__( 'Order', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'desc',
				'options' => [
					'asc'  => __( 'ASC', 'tastyc-plugin' ),
					'desc' => __( 'DESC', 'tastyc-plugin' ),
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'pagination_tab',
			[
				'label' => esc_html__( 'Pagination', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'pagination',
			[
				'label'       => esc_html__( 'Pagination Type', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'pages',
				'options' => [
					'pages' => __( 'Pages', 'tastyc-plugin' ),
					'button' => __( 'Button', 'tastyc-plugin' ),
					'no' => __( 'No', 'tastyc-plugin' ),
				],
			]
		);

		$this->add_control(
			'more_text',
			[
				'label'       => esc_html__( 'More Text', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter text', 'tastyc-plugin' ),
				'condition' => [
		            'pagination' => 'button'
		        ],
			]
		);

		$this->add_control(
			'more_btn_txt',
			[
				'label'       => esc_html__( 'Button (title)', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter button', 'tastyc-plugin' ),
				'default'     => esc_html__( 'All Posts', 'tastyc-plugin' ),
				'condition' => [
		            'pagination' => 'button'
		        ],
			]
		);

		$this->add_control(
			'more_btn_link',
			[
				'label'       => esc_html__( 'Button (link)', 'tastyc-plugin' ),
				'type'        => Controls_Manager::URL,
				'show_external' => true,
				'condition' => [
		            'pagination' => 'button'
		        ],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'settings_tab',
			[
				'label' => esc_html__( 'Settings', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'layout',
			[
				'label'       => esc_html__( 'Layout', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'column-3',
				'options' => [
					'column-1'  => __( '1 Columns', 'tastyc-plugin' ),
					'column-2'  => __( '2 Columns', 'tastyc-plugin' ),
					'column-3' => __( '3 Columns', 'tastyc-plugin' ),
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'title_styling',
			[
				'label'     => esc_html__( 'Title', 'tastyc-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'subtitle_color',
			[
				'label'     => esc_html__( 'Subtitle Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .row .tst-suptitle' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'subtitle_typography',
				'label'     => esc_html__( 'Subtitle Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .row .tst-suptitle',
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__( 'Title Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .row .tst-title--h' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'label'     => esc_html__( 'Title Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .row .tst-title--h',
			]
		);

		$this->add_control(
			'desc_color',
			[
				'label'     => esc_html__( 'Description Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .row .tst-title--desc' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'desc_typography',
				'label'     => esc_html__( 'Description Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .row .tst-title--desc',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'items_styling',
			[
				'label'     => esc_html__( 'Items', 'tastyc-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

    $this->add_control(
			'item_bg_color',
			[
				'label'     => esc_html__( 'Item BG Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-blog-card' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'item_category_color',
			[
				'label'     => esc_html__( 'Category Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-blog-card .tst-cover-frame .tst-card-badge' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'item_category_bgcolor',
			[
				'label'     => esc_html__( 'Category BG Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-blog-card .tst-cover-frame .tst-card-badge' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_category_typography',
				'label'     => esc_html__( 'Category Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-blog-card .tst-cover-frame .tst-card-badge',
			]
		);

		$this->add_control(
			'item_title_color',
			[
				'label'     => esc_html__( 'Title Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-blog-card h5 a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_title_typography',
				'label'     => esc_html__( 'Title Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-blog-card h5 a',
			]
		);

		$this->add_control(
			'item_desc_color',
			[
				'label'     => esc_html__( 'Description Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-blog-card .tst-text' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_desc_typography',
				'label'     => esc_html__( 'Description Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-blog-card .tst-text',
			]
		);

		$this->add_control(
			'item_author_color',
			[
				'label'     => esc_html__( 'Author Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-blog-card h6' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_author_typography',
				'label'     => esc_html__( 'Author Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-blog-card h6',
			]
		);

		$this->add_control(
			'item_date_color',
			[
				'label'     => esc_html__( 'Date Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-blog-card .tst-date' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'item_date_bgcolor',
			[
				'label'     => esc_html__( 'Date BG Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-blog-card .tst-date' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_date_typography',
				'label'     => esc_html__( 'Date Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-blog-card .tst-date',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'pagination_styling',
			[
				'label'     => esc_html__( 'Pagination', 'tastyc-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'pagination_color',
			[
				'label'     => esc_html__( 'Pagination Links Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-pagination a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'pagination_typography',
				'label'     => esc_html__( 'Pagination Links Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-pagination',
			]
		);

		$this->add_control(
			'more_text_color',
			[
				'label'     => esc_html__( 'More Text Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-read-more .tst-text' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'more_text_typography',
				'label'     => esc_html__( 'More Text Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-read-more .tst-text',
			]
		);

		$this->add_control(
			'more_button_color',
			[
				'label'     => esc_html__( 'More Button Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-read-more .tst-btn' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'more_button_bgcolor',
			[
				'label'     => esc_html__( 'More Button BG Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-read-more .tst-btn' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'more_button_typography',
				'label'     => esc_html__( 'More Button Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-read-more .tst-btn',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render Categories List.
	 *
	 * @since 1.0
	 */
	protected function get_blog_categories() {
		$categories = [];

		$args = array(
			'type'			=> 'post',
			'child_of'		=> 0,
			'parent'		=> '',
			'orderby'		=> 'name',
			'order'			=> 'DESC',
			'hide_empty'	=> 1,
			'hierarchical'	=> 1,
			'taxonomy'		=> 'category',
			'pad_counts'	=> false
		);

		$blog_categories = get_categories( $args );

		foreach ( $blog_categories as $category ) {
			$categories[$category->term_id] = $category->name;
		}

		return $categories;
	}

	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$this->add_inline_editing_attributes( 'title', 'basic' );
		$this->add_inline_editing_attributes( 'subtitle', 'basic' );
		$this->add_inline_editing_attributes( 'description', 'basic' );
		$this->add_inline_editing_attributes( 'more_btn_txt', 'none' );
		$this->add_inline_editing_attributes( 'more_text', 'basic' );

		$paged = ( get_query_var( 'paged' ) ) ? absint( get_query_var( 'paged' ) ) : 1;
		$page_id = get_the_ID();

		if ( $settings['source'] == 'all' ) {
			$cat_ids = '';
		} else {
			$cat_ids = $settings['source_categories'];
		}

		$cat_args = array(
			'type'			=> 'post',
			'child_of'		=> 0,
			'parent'		=> '',
			'orderby'		=> 'name',
			'order'			=> 'DESC',
			'hide_empty'	=> 1,
			'hierarchical'	=> 1,
			'taxonomy'		=> 'category',
			'pad_counts'	=> false,
			'include'		=> $cat_ids
		);

		$bp_categories = get_categories( $cat_args );

		$args = array(
			'post_type'			=> 'post',
			'post_status'		=> 'publish',
			'orderby'			=> $settings['sort'],
			'order'				=> $settings['order'],
			'posts_per_page'	=> $settings['limit'],
			'paged' 			=> $paged
		);

		if( $settings['source'] == 'categories' ) {
			$tax_array = array(
				array(
					'taxonomy' => 'category',
					'field'    => 'id',
					'terms'    => $cat_ids
				)
			);

			$args += array('tax_query' => $tax_array);
		}

		$q = new \WP_Query( $args );

		?>

		<!-- container -->
		<div class="container">

		<div class="row align-items-center">
		  <?php if ( $settings['title'] || $settings['subtitle'] || $settings['description'] ) : ?>
          <div class="col-lg-12">

            <!-- title -->
            <div class="text-center">
              <?php if ( $settings['subtitle'] ) : ?>
              <div class="tst-suptitle tst-suptitle-center tst-mb-15">
              	<span <?php echo $this->get_render_attribute_string( 'subtitle' ); ?>>
		        	<?php echo wp_kses_post( $settings['subtitle'] ); ?>
		        </span>
              </div>
              <?php endif; ?>
              <?php if ( $settings['title'] ) : ?>
              <<?php echo esc_attr( $settings['title_tag'] ); ?> class="tst-title--h tst-mb-30">
              	<span <?php echo $this->get_render_attribute_string( 'title' ); ?>>
		        	<?php echo wp_kses_post( $settings['title'] ); ?>
		        </span>
              </<?php echo esc_attr( $settings['title_tag'] ); ?>>
              <?php endif; ?>
              <?php if ( $settings['description'] ) : ?>
              <p class="tst-text tst-title--desc tst-mb-60">
              	<span <?php echo $this->get_render_attribute_string( 'description' ); ?>>
		        	<?php echo wp_kses_post( $settings['description'] ); ?>
		        </span>
              </p>
              <?php endif; ?>
            </div>
            <!-- title end-->

          </div>
          <?php endif; ?>
          <?php if ( $q->have_posts() ) : ?>
          <?php while ( $q->have_posts() ) : $q->the_post(); ?>
          <div class="<?php if ( $settings['layout'] == 'column-1' ) : ?>col-lg-12<?php endif; ?><?php if ( $settings['layout'] == 'column-2' ) : ?>col-lg-6<?php endif; ?><?php if ( $settings['layout'] == 'column-3' ) : ?>col-lg-4<?php endif; ?>">
            <?php
            set_query_var( 'layout', $settings['layout'] );
            get_template_part( 'template-parts/content', 'grid' ); ?>
          </div>
          <?php endwhile; ?>

          <?php if ( $settings['pagination'] == 'pages' ) : ?>
		  <div class="col-lg-12">
		    <!-- pagination -->
		    <div class="tst-a tst-pagination">
		      	<?php
					$big = 999999999; // need an unlikely integer

					echo paginate_links( array(
						'base' => str_replace( $big, '%#%', esc_url( get_pagenum_link( $big ) ) ),
						'format' => '?paged=%#%',
						'current' => max( 1, get_query_var('paged') ),
						'total' => $q->max_num_pages,
						'prev_text' => esc_html__( 'Prev', 'tastyc-plugin' ),
						'next_text' => esc_html__( 'Next', 'tastyc-plugin' ),
					) );
				?>
		    </div>
		    <!-- pagination end -->
		  </div>
		  <?php endif; ?>

		  <?php if ( $settings['pagination'] == 'button' ) : ?>
		  <div class="col-lg-12">
            <!-- read more -->
            <div class="tst-read-more">
              <?php if ( $settings['more_text'] ) : ?>
              <div class="tst-text">
              	<span <?php echo $this->get_render_attribute_string( 'more_text' ); ?>>
		        	<?php echo wp_kses_post( $settings['more_text'] ); ?>
		        </span>
              </div>
              <?php endif; ?>
              <a<?php if ( $settings['more_btn_link'] ) : ?><?php if ( $settings['more_btn_link']['is_external'] ) : ?> target="_blank"<?php endif; ?><?php if ( $settings['more_btn_link']['nofollow'] ) : ?> rel="nofollow"<?php endif; ?> href="<?php echo esc_url( $settings['more_btn_link']['url'] ); ?>"<?php endif; ?> class="tst-btn tst-anima-link">
              	<span <?php echo $this->get_render_attribute_string( 'more_btn_txt' ); ?>>
		        	<?php echo wp_kses_post( $settings['more_btn_txt'] ); ?>
		        </span>
              </a>
            </div>
            <!-- read more end -->
          </div>
          <?php endif; ?>
          <?php else : ?>
		  <div class="col-lg-12">
				<?php get_template_part( 'template-parts/content', 'none' ); ?>
		  </div>
		  <?php endif; wp_reset_postdata(); ?>
        </div>

        <div class="tst-spacer"></div>

    	</div>

		<?php
	}
}

Plugin::instance()->widgets_manager->register( new Tastyc_Blog_Grid_Widget() );
